/* /////////////////////////////////////////////////////////////////////////////
 * File:        std/dtl/range/intrange.d
 *
 * Purpose:     IntRange class.
 *
 * Created      21st January 2004
 * Updated:     31st July 2004
 *
 * www:         http://www.synesis.com.au/software/
 *
 * Copyright (C) 2004 by Matthew Wilson
 *
 * This software is provided 'as-is', without any express or implied warranty.
 * In no event will the authors be held liable for any damages arising from the
 * use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not claim
 * that you wrote the original software. If you use this software in a product,
 * an acknowledgment in the product documentation would be appreciated but is
 * not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must not be
 * misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source distribution. 
 *
 * ////////////////////////////////////////////////////////////////////////// */


/** \file std/dtl/range/intrange.d IntRange class */

/* ////////////////////////////////////////////////////////////////////////// */

module std.dtl.range.intrange;

/* /////////////////////////////////////////////////////////////////////////////
 * Imports
 */

import std.dtl.range.categories;
import std.dtl.range.ranges;

/* /////////////////////////////////////////////////////////////////////////////
 * Classes
 */

template IntegralRange(T) { class IntegralRange
	: public SubscriptableRange!(T)
{
/// \name Construction
/// @{
public:
	alias	SubscriptableRange!(T)	range_type;
	alias	T						value_type;
	alias	IntegralRange			class_type;
/// @}

/// \name Construction
/// @{
public:
	this(value_type first, value_type last, value_type increment)
	in
	{
        assert(m_position == m_last || (increment > 0 && m_last > m_position) || (m_increment < 0 && m_last < m_position));
        assert(0 == ((m_last - m_position) % increment));
	}
	body
	{
		m_position	=	first;
		m_last		=	last;
		m_increment	=	increment;
	}
/// @}

/// \name Range methods
/// @{
public:
	class_type dup()
	{
		return new class_type(m_position, m_last, m_increment);
	}
/// @}

/// \name Notional Range methods
/// @{
public:
    bool is_open()
	{
		return m_position != m_last;
	}

    value_type current()
	{
		return m_position;
	}

    void advance()
	in
	{
        assert((m_increment > 0 && m_position < m_last) || (m_increment < 0 && m_position > m_last));
	}
	body
	{
        m_position += m_increment;
	}
/// @}

/// \name Subscriptable Range methods
/// @{
public:
	;
 	uint length()
	{
		if(m_last < m_position)
		{
			return ((m_position - m_last) - (m_increment + 1)) / -m_increment;
		}
		else
		{
			return ((m_last - m_position) + (m_increment - 1)) / m_increment;
		}
	}

	value_type opIndex(index_type index)
	in
	{
        assert((m_position + index * m_increment) == m_last || (m_increment > 0 && m_last > (m_position + index * m_increment)) || (m_increment < 0 && m_last < (m_position + index * m_increment)));
	}
	body
	{
		return m_position + index * m_increment;
	}

    void advance(difference_type increment)
	in
	{
        assert((m_position + increment * m_increment) == m_last || (m_increment > 0 && m_last > (m_position + increment * m_increment)) || (m_increment < 0 && m_last < (m_position + increment * m_increment)));
	}
	body
	{
		m_position += increment * m_increment;
	}
/// @}

/// \name foreach enumeration
/// @{
public:
	int opApply(int delegate(inout value_type v) dg)
	{
		int	res = 0;

		if(m_last < m_position)
		{
			for(value_type i = m_position; i > m_last; i += m_increment)
			{
				res = dg(i);

				if(0 != res)
				{
					break;
				}
			}
		}
		else
		{
			for(value_type i = m_position; i < m_last; i += m_increment)
			{
				res = dg(i);

				if(0 != res)
				{
					break;
				}
			}
		}

		return res;
	}
/// @}

private:
	invariant
	{
		assert(	m_position == m_last ||
				(	(m_position < m_last) && 
					m_increment > 0 ) ||
				(	(m_last < m_position) &&
					m_increment < 0));
	}

private:
	value_type	m_current;
	value_type	m_position;
	value_type	m_last;
	value_type	m_increment;
}}


/+
int main(char[][] args)
{
	const int min	=	100;
	const int max	=	10;
	const int step	=	-5;

	IntRange	r1	=	new IntRange(max, min, -step);
	IntRange	r2	=	new IntRange(min, max, step);
	int			l1	=	r1.length;
	int			l2	=	r2.length;

	char[][char[]]	names;

	names["Matthew"] = "Wilson";
	names["Benjamin"] = "Bunny";

	printf("%d\n", l1);
	printf("%d;%d;%d: %d\n", min, max, step, l1);

	printf("Testing temp range:\n");
	foreach(int i; new IntRange(0, 10, +1))
	{
		printf("%d ", i);
	}
	printf("\n");
	printf("\n");

	foreach(int i; r1)
	{
		printf("%d ", i);
	}
	printf("\n");
	printf("\n");

//printf("Using display():\n");
//r1.display();

	foreach(char[] n; names)
	{
		printf("%.*s ", n);
	}
	printf("\n");
	printf("\n");

	return 0;
}
+/

/* ////////////////////////////////////////////////////////////////////////// */
