/* /////////////////////////////////////////////////////////////////////////////
 * File:        std/dtl/range/filters.d
 *
 * Purpose:     Range filter classes - comparison, matching, transformation
 *
 * Created      15th July 2004
 * Updated:     20th August 2004
 *
 * www:         http://www.synesis.com.au/software/
 *
 * Copyright (C) 2004 by Matthew Wilson
 *
 * This software is provided 'as-is', without any express or implied warranty.
 * In no event will the authors be held liable for any damages arising from the
 * use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not claim
 * that you wrote the original software. If you use this software in a product,
 * an acknowledgment in the product documentation would be appreciated but is
 * not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must not be
 * misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source distribution. 
 *
 * ////////////////////////////////////////////////////////////////////////// */


/** \file std/dtl/range/filters.d Range filter classes */

/* ////////////////////////////////////////////////////////////////////////// */

module std.dtl.range.filters;

/* /////////////////////////////////////////////////////////////////////////////
 * Imports
 */

import std.dtl.common;
import std.dtl.functions.functions;
import std.dtl.functions.predicates;
import std.dtl.range.categories;
import std.dtl.range.ranges;

/+
    range       -   transform           -   uses TransformedRange
    bool        -   detect              -   returns scalar
    range       -   entries             -   dunno yet!
    bool        -   contains            -   returns scalar
    value_type  -   min / minWith       -   returns scalar
    value_type  -   max / maxWith       -   returns scalar
    range       -   reject              -   uses MatchedRange
    range       -   select              -   uses MatchedRange
    range       -   sort / sortWith     -   dunno yet!
+/

/* ////////////////////////////////////////////////////////////////////////// */

/// \brief Matched filtered range
class MatchedRange(R, F)
    : public NotionalRange!(R.value_type)
{
public:
    /// The filtered range type
    alias   R                                                   				filtered_range_type;
    /// The functor type
    alias   F                                                   				filter_type;
    /// The value type
    alias   R.value_type                                        				value_type;
    /// The current instantion of this type
    alias   MatchedRange                                        				class_type;
    /// The range type of this class
    alias   class_type                                          				range_type;
public:
    /// The comparison delegate
    alias   int delegate(in value_type v1, in value_type v2)    				compare_delegate_type;
    /// The comparison function
    alias   int function(in value_type v1, in value_type v2)    				compare_function_type;
    /// The match delegate
    alias   bool delegate(in value_type value)                  				match_delegate_type;
    /// The match function
    alias   bool function(in value_type value)                  				match_function_type;
    /// The transformation delegate
    alias   value_type delegate(in value_type value)            				transform_delegate_type;
    /// The transformation function
    alias   value_type function(in value_type value)							transform_function_type;

private:
//  alias FunctionPredicate!(compare_function_type, value_type)                 compare_fn_pred_type;
//  alias DelegatePredicate!(compare_delegate_type, value_type)                 compare_dg_pred_type;
    alias FunctionPredicate!(match_function_type, value_type)                   match_fn_pred_type;
    alias DelegatePredicate!(match_delegate_type, value_type)                   match_dg_pred_type;
    alias FunctionFunction!(transform_function_type, value_type, value_type)    transform_fn_pred_type;
    alias DelegateFunction!(transform_delegate_type, value_type, value_type)    transform_dg_pred_type;

public:
    this(filtered_range_type r, filter_type f)
    {
//printf("MatchedRange::MatchedRange(%p, %p)\n", r, f);

        m_r = r.dup();
        m_f = f;

        for(; m_r.is_open() && !m_f(m_r.current()); m_r.advance())
        {}
    }
/*     this(filtered_range_type r)
    {
        m_r = r.dup();
        m_f = new filter_type();

        for(; m_r.is_open() && !m_f(m_r.current()); m_r.advance())
        {}
    }
 */
/// \name Range methods
/// @{
public:
    class_type dup()
    {
        return new class_type(m_r.dup(), m_f);
    }

    range_type opSlice()
    {
        return dup();
    }
/// @}

/// \name Notional Range methods
/// @{
public:
    bool is_open()
    {
        return m_r.is_open();
    }

    value_type current()
    {
        return m_r.current();
    }

    void advance()
    {
        for(m_r.advance(); m_r.is_open() && !m_f(m_r.current()); m_r.advance())
        {}
    }
/// @}

/// \name Transformation methods
/// @{
public:
    TransformedRange0!(range_type, transform_fn_pred_type, value_type) transform(transform_function_type d)
    {
        return new TransformedRange0!(range_type, transform_fn_pred_type, value_type)(opSlice(), new transform_fn_pred_type(d));
    }
    TransformedRange0!(range_type, transform_dg_pred_type, value_type) transform(transform_delegate_type d)
    {
        return new TransformedRange0!(range_type, transform_dg_pred_type, value_type)(opSlice(), new transform_dg_pred_type(d));
    }

    template transformWith0(F, T2 = value_type) { final TransformedRange0!(range_type, F, T2) transformWith0(F f = new F())
    {
        return new TransformedRange0!(range_type, F, T2)(opSlice(), f);
    }}
    template transformWith1(F, T2 = value_type) { final TransformedRange0!(range_type, F, T2) transformWith1(F f)
    {
        return new TransformedRange0!(range_type, F, T2)(opSlice(), f);
    }}
/// @}

/// \name Selection methods
/// @{
public:
    /// \brief Returns a range representing all elements
    /// for which the delegate \c d returns false.
    /// \note This is the opposite of select()
    MatchedRange0!(range_type, Not!(match_fn_pred_type)) reject(match_function_type d)
    {
        return new MatchedRange0!(range_type, Not!(match_fn_pred_type))(opSlice(), new Not!(match_fn_pred_type)(new match_fn_pred_type(d)));
    }

    MatchedRange0!(range_type, Not!(match_dg_pred_type)) reject(match_delegate_type d)
    {
        return new MatchedRange0!(range_type, Not!(match_dg_pred_type))(opSlice(), new Not!(match_dg_pred_type)(new match_dg_pred_type(d)));
    }

    /// \brief Returns a range representing all elements
    /// for which the predicate \c p returns false.
    /// \note This is the opposite of select()
    template reject0(P) { final MatchedRange0!(range_type, Not!(P)) reject0()
    {
        return new MatchedRange0!(range_type, Not!(P))(opSlice(), new Not!(P)(new P()));
    }}
    template reject1(P) { final MatchedRange0!(range_type, Not!(P)) reject1(P p)
    {
        return new MatchedRange0!(range_type, Not!(P))(opSlice(), new Not!(P)(p));
    }}

    /// \brief Returns a range representing all elements
    /// for which the delegate \c d returns true.
    /// \note This is the opposite of select()
    MatchedRange0!(range_type, match_fn_pred_type) select(match_function_type d)
    {
        return new MatchedRange0!(range_type, match_fn_pred_type)(opSlice(), new match_fn_pred_type(d));
    }
    MatchedRange0!(range_type, match_dg_pred_type) select(match_delegate_type d)
    {
        return new MatchedRange0!(range_type, match_dg_pred_type)(opSlice(), new match_dg_pred_type(d));
    }
    /// \brief Returns a range representing all elements
    /// for which the function \c f returns true
    /// \note This is the opposite of reject()
    template select0(F) { final MatchedRange0!(range_type, F) select0()
    {
        return new MatchedRange0!(range_type, F)(opSlice(), new F());
    }}
    template select1(F) { final MatchedRange0!(range_type, F) select1(F f)
    {
        return new MatchedRange0!(range_type, F)(opSlice(), f);
    }}
/// @}

//	detect, etc. call a dtl-global shim. This then either selects from (e.g. random-access) range tags, or does the canonical (less efficient) version

/// \name foreach enumeration
/// @{
public:
    int opApply(int delegate(inout value_type v) dg)
    {
        int         res =   0;
        class_type  r   =   dup();

        for(; r.is_open(); r.advance())
        {
            value_type  val =   r.current();

            res = dg(val);

            if(0 != res)
            {
                break;
            }
        }

        return res;
    }
/// @}

private:
    filtered_range_type m_r;
    filter_type         m_f;
}

/* ////////////////////////////////////////////////////////////////////////// */

/// \brief Matched range
class MatchedRange0(R, F)
    : public NotionalRange!(R.value_type)
{
public:
    alias   class_type                                              range_type;

    alias   R                                                       filtered_range_type;
    alias   F                                                       filter_type;
    alias   filtered_range_type.value_type                          value_type;

    alias   MatchedRange0											class_type;
public:
    alias   int delegate(in value_type v1, in value_type v2)        compare_delegate_type;
    alias   int function(in value_type v1, in value_type v2)        compare_function_type;
    alias   bool delegate(in value_type value)                      match_delegate_type;
    alias   bool function(in value_type value)                      match_function_type;
    alias   value_type delegate(in value_type value)                transform_delegate_type;
    alias   value_type function(in value_type value)                transform_function_type;

private:
    alias FunctionPredicate!(match_function_type, value_type)       match_fn_pred_type;
    alias DelegatePredicate!(match_delegate_type, value_type)       match_dg_pred_type;

public:
    this(filtered_range_type r, filter_type f)
    {
        m_r = r.dup();
        m_f = f;

        for(; m_r.is_open() && !m_f(m_r.current()); m_r.advance())
        {}
    }
/*     this(filtered_range_type r)
    {
        m_r = r.dup();
        m_f = new filter_type();

        for(; m_r.is_open() && !m_f(m_r.current()); m_r.advance())
        {}
    }
 */
/// \name Range methods
/// @{
public:
    class_type dup()
    {
        return new class_type(m_r.dup(), m_f);
    }

    range_type opSlice()
    {
        return dup();
    }
/// @}

/// \name Notional Range methods
/// @{
public:
    bool is_open()
    {
        return m_r.is_open();
    }

    value_type current()
    {
        return m_r.current();
    }

    void advance()
    {
        for(m_r.advance(); m_r.is_open() && !m_f(m_r.current()); m_r.advance())
        {}
    }
/// @}

/+
    /// \brief Returns a range representing all elements
    /// for which the delegate \c d returns false.
    /// \note This is the opposite of select()
    MatchedRange1!(range_type, Not!(match_fn_pred_type)) reject(match_function_type d)
    {
        return new MatchedRange1!(range_type, Not!(match_fn_pred_type))(opSlice(), new Not!(match_fn_pred_type)(new match_fn_pred_type(d)));
    }

    MatchedRange1!(range_type, Not!(match_dg_pred_type)) reject(match_delegate_type d)
    {
        return new MatchedRange1!(range_type, Not!(match_dg_pred_type))(opSlice(), new Not!(match_dg_pred_type)(new match_dg_pred_type(d)));
    }

    /// \brief Returns a range representing all elements
    /// for which the predicate \c p returns false.
    /// \note This is the opposite of select()
    template reject0(P) { final MatchedRange1!(range_type, Not!(P)) reject0()
    {
        return new MatchedRange1!(range_type, Not!(P))(opSlice(), new Not!(P)(new P()));
    }}
    template reject1(P) { final MatchedRange1!(range_type, Not!(P)) reject1(P p)
    {
        return new MatchedRange1!(range_type, Not!(P))(opSlice(), new Not!(P)(p));
    }}

    /// \brief Returns a range representing all elements
    /// for which the delegate \c d returns true.
    /// \note This is the opposite of select()
    MatchedRange1!(range_type, match_fn_pred_type) select(match_function_type d)
    {
        return new MatchedRange1!(range_type, match_fn_pred_type)(opSlice(), new match_fn_pred_type(d));
    }
    MatchedRange1!(range_type, match_dg_pred_type) select(match_delegate_type d)
    {
        return new MatchedRange1!(range_type, match_dg_pred_type)(opSlice(), new match_dg_pred_type(d));
    }
    /// \brief Returns a range representing all elements
    /// for which the function \c f returns true
    /// \note This is the opposite of reject()
    template select0(F) { final MatchedRange1!(range_type, F) select0()
    {
        return new MatchedRange1!(range_type, F)(opSlice(), new F());
    }}
    template select1(F) { final MatchedRange1!(range_type, F) select1(F f)
    {
        return new MatchedRange1!(range_type, F)(opSlice(), f);
    }}
+/

/// \name foreach enumeration
/// @{
public:
    int opApply(int delegate(inout value_type v) dg)
    {
        int         res =   0;
        class_type  r   =   dup();

        for(; r.is_open(); r.advance())
        {
            value_type  val =   r.current();

            res = dg(val);

            if(0 != res)
            {
                break;
            }
        }

        return res;
    }
/// @}

private:
    filtered_range_type m_r;
    filter_type         m_f;
}

/* ////////////////////////////////////////////////////////////////////////// */

/+
/// \brief Matched range
class MatchedRange1(R, F)
    : public NotionalRange!(R.value_type)
{
public:
    alias   R                                                       filtered_range_type;
    alias   F                                                       filter_type;
    alias   filtered_range_type.value_type                          value_type;
    alias   MatchedRange1                                           class_type;
    alias   class_type                                              range_type;
public:
    alias   int delegate(in value_type v1, in value_type v2)        compare_delegate_type;
    alias   int function(in value_type v1, in value_type v2)        compare_function_type;
    alias   bool delegate(in value_type value)                      match_delegate_type;
    alias   bool function(in value_type value)                      match_function_type;
    alias   value_type delegate(in value_type value)                transform_delegate_type;
    alias   value_type function(in value_type value)                transform_function_type;

private:
    alias FunctionPredicate!(match_function_type, value_type)       match_fn_pred_type;
    alias DelegatePredicate!(match_delegate_type, value_type)       match_dg_pred_type;

public:
    this(filtered_range_type r, filter_type f)
    {
        m_r = r.dup();
        m_f = f;

        for(; m_r.is_open() && !m_f(m_r.current()); m_r.advance())
        {}
    }
/*     this(filtered_range_type r)
    {
        m_r = r.dup();
        m_f = new filter_type();

        for(; m_r.is_open() && !m_f(m_r.current()); m_r.advance())
        {}
    }
 */
/// \name Range methods
/// @{
public:
    class_type dup()
    {
        return new class_type(m_r.dup(), m_f);
    }

    range_type opSlice()
    {
        return dup();
    }
/// @}

/// \name Notional Range methods
/// @{
public:
    bool is_open()
    {
        return m_r.is_open();
    }

    value_type current()
    {
        return m_r.current();
    }

    void advance()
    {
        for(m_r.advance(); m_r.is_open() && !m_f(m_r.current()); m_r.advance())
        {}
    }
/// @}

/+
    MatchedRange1!(range_type, Not!(match_fn_pred_type)) reject(match_function_type d)
    {
        return new MatchedRange1!(range_type, Not!(match_fn_pred_type))(opSlice(), new Not!(match_fn_pred_type)(new match_fn_pred_type(d)));
    }

    MatchedRange1!(range_type, Not!(match_dg_pred_type)) reject(match_delegate_type d)
    {
        return new MatchedRange1!(range_type, Not!(match_dg_pred_type))(opSlice(), new Not!(match_dg_pred_type)(new match_dg_pred_type(d)));
    }
+/

/// \name foreach enumeration
/// @{
public:
    int opApply(int delegate(inout value_type v) dg)
    {
        int         res =   0;
        class_type  r   =   dup();

        for(; r.is_open(); r.advance())
        {
            value_type  val =   r.current();

            res = dg(val);

            if(0 != res)
            {
                break;
            }
        }

        return res;
    }
/// @}

private:
    filtered_range_type m_r;
    filter_type         m_f;
}
+/

/* ////////////////////////////////////////////////////////////////////////// */

/// \brief Transformed range
class TransformedRange(R, F, T)
    : public NotionalRange!(T)
{
public:
    /// The filtered range type
    alias   R                                                   				filtered_range_type;
    /// The functor type
    alias   F                                                   				filter_type;
    /// The value type
    alias   T                                                   				value_type;
    /// The current instantion of this type
    alias   TransformedRange                                    				class_type;
    /// The range type of this class
    alias   class_type                                          				range_type;
public:
    /// The comparison delegate
    alias   int delegate(in value_type v1, in value_type v2)    				compare_delegate_type;
    /// The comparison function
    alias   int function(in value_type v1, in value_type v2)    				compare_function_type;
    /// The match delegate
    alias   bool delegate(in value_type value)                  				match_delegate_type;
    /// The match function
    alias   bool function(in value_type value)                  				match_function_type;
    /// The transformation delegate
    alias   value_type delegate(in value_type value)            				transform_delegate_type;
    /// The transformation function
    alias   value_type function(in value_type value)            				transform_function_type;
private:
//  alias FunctionPredicate!(compare_function_type, value_type)                 compare_fn_pred_type;
//  alias DelegatePredicate!(compare_delegate_type, value_type)                 compare_dg_pred_type;
    alias FunctionPredicate!(match_function_type, value_type)                   match_fn_pred_type;
    alias DelegatePredicate!(match_delegate_type, value_type)                   match_dg_pred_type;
    alias FunctionFunction!(transform_function_type, value_type, value_type)    transform_fn_pred_type;
    alias DelegateFunction!(transform_delegate_type, value_type, value_type)    transform_dg_pred_type;


public:
    this(filtered_range_type r, filter_type f)
    {
        m_r = r.dup;
        m_f = f;
    }
/+
    this(filtered_range_type r)
    {
        m_r = r.dup;
        m_f = new filter_type();
    }
+/

/// \name Range methods
/// @{
public:
    class_type dup()
    {
        return new class_type(m_r.dup(), m_f);
    }

    range_type opSlice()
    {
        return dup();
    }
/// @}

/// \name Notional Range methods
/// @{
public:
    bool is_open()
    {
        return m_r.is_open();
    }

    value_type current()
    {
        return m_f(m_r.current());
    }

    void advance()
    {
        return m_r.advance();
    }
/// @}

    TransformedRange0!(range_type, transform_fn_pred_type, value_type) transform(transform_function_type d)
    {
        return new TransformedRange0!(range_type, transform_fn_pred_type, value_type)(opSlice(), new transform_fn_pred_type(d));
    }
    TransformedRange0!(range_type, transform_dg_pred_type, value_type) transform(transform_delegate_type d)
    {
        return new TransformedRange0!(range_type, transform_dg_pred_type, value_type)(opSlice(), new transform_dg_pred_type(d));
    }

    template transformWith0(F, T2 = value_type) { final TransformedRange0!(range_type, F, T2) transformWith0(F f = new F())
    {
        return new TransformedRange0!(range_type, F, T2)(opSlice(), f);
    }}
    template transformWith1(F, T2 = value_type) { final TransformedRange0!(range_type, F, T2) transformWith1(F f)
    {
        return new TransformedRange0!(range_type, F, T2)(opSlice(), f);
    }}

    /// \brief Returns a range representing all elements
    /// for which the delegate \c d returns false.
    /// \note This is the opposite of select()
    MatchedRange0!(range_type, Not!(match_fn_pred_type)) reject(match_function_type d)
    {
        return new MatchedRange0!(range_type, Not!(match_fn_pred_type))(opSlice(), new Not!(match_fn_pred_type)(new match_fn_pred_type(d)));
    }

    MatchedRange0!(range_type, Not!(match_dg_pred_type)) reject(match_delegate_type d)
    {
        return new MatchedRange0!(range_type, Not!(match_dg_pred_type))(opSlice(), new Not!(match_dg_pred_type)(new match_dg_pred_type(d)));
    }

    /// \brief Returns a range representing all elements
    /// for which the predicate \c p returns false.
    /// \note This is the opposite of select()
    template reject0(P) { final MatchedRange0!(range_type, Not!(P)) reject0()
    {
        return new MatchedRange0!(range_type, Not!(P))(opSlice(), new Not!(P)(new P()));
    }}
    template reject1(P) { final MatchedRange0!(range_type, Not!(P)) reject1(P p)
    {
        return new MatchedRange0!(range_type, Not!(P))(opSlice(), new Not!(P)(p));
    }}

    /// \brief Returns a range representing all elements
    /// for which the delegate \c d returns true.
    /// \note This is the opposite of select()
    MatchedRange0!(range_type, match_fn_pred_type) select(match_function_type d)
    {
        return new MatchedRange0!(range_type, match_fn_pred_type)(opSlice(), new match_fn_pred_type(d));
    }
    MatchedRange0!(range_type, match_dg_pred_type) select(match_delegate_type d)
    {
        return new MatchedRange0!(range_type, match_dg_pred_type)(opSlice(), new match_dg_pred_type(d));
    }
    /// \brief Returns a range representing all elements
    /// for which the function \c f returns true
    /// \note This is the opposite of reject()
    template select0(F) { final MatchedRange0!(range_type, F) select0()
    {
        return new MatchedRange0!(range_type, F)(opSlice(), new F());
    }}
    template select1(F) { final MatchedRange0!(range_type, F) select1(F f)
    {
        return new MatchedRange0!(range_type, F)(opSlice(), f);
    }}

/// \name foreach enumeration
/// @{
public:
    int opApply(int delegate(inout value_type v) dg)
    {
        int         res =   0;
        class_type  r   =   dup();

        for(; r.is_open(); r.advance())
        {
            value_type  val =   r.current();

            res = dg(val);

            if(0 != res)
            {
                break;
            }
        }

        return res;
    }
/// @}

private:
    filtered_range_type m_r;
    filter_type         m_f;
}

/* ////////////////////////////////////////////////////////////////////////// */

/// \brief Transformed range
class TransformedRange0(R, F, T)
    : public NotionalRange!(T)
{
public:
    alias   R                                   filtered_range_type;
    alias   F                                   filter_type;
    alias   T                                   value_type;
    alias   TransformedRange0                   class_type;
    alias   class_type                          range_type;

public:
    this(filtered_range_type r, filter_type f)
    {
        m_r = r.dup();
        m_f = f;
    }
/+
    this(filtered_range_type r)
    {
        m_r = r.dup();
        m_f = new filter_type();
    }
+/

/// \name Range methods
/// @{
public:
    class_type dup()
    {
        return new class_type(m_r.dup(), m_f);
    }

    range_type opSlice()
    {
        return dup();
    }
/// @}

/// \name Notional Range methods
/// @{
public:
    bool is_open()
    {
        return m_r.is_open();
    }

    value_type current()
    {
        return m_f(m_r.current());
    }

    void advance()
    {
        return m_r.advance();
    }
/// @}

/// \name foreach enumeration
/// @{
public:
    int opApply(int delegate(inout value_type v) dg)
    {
        int         res =   0;
        class_type  r   =   dup();

        for(; r.is_open(); r.advance())
        {
            value_type  val =   r.current();

            res = dg(val);

            if(0 != res)
            {
                break;
            }
        }

        return res;
    }
/// @}

private:
    filtered_range_type m_r;
    filter_type         m_f;
}

/* ////////////////////////////////////////////////////////////////////////// */
