/* /////////////////////////////////////////////////////////////////////////////
 * File:        std/dtl/vector.d
 *
 * Purpose:     Vector template class.
 *
 * Created      21st January 2004
 * Updated:     20th August 2004
 *
 * www:         http://www.synesis.com.au/software/
 *
 * Copyright (C) 2004 by Matthew Wilson
 *
 * This software is provided 'as-is', without any express or implied warranty.
 * In no event will the authors be held liable for any damages arising from the
 * use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not claim
 * that you wrote the original software. If you use this software in a product,
 * an acknowledgment in the product documentation would be appreciated but is
 * not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must not be
 * misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source distribution. 
 *
 * ////////////////////////////////////////////////////////////////////////// */


/** \file std/dtl/vector.d Vector template class */

/* ////////////////////////////////////////////////////////////////////////// */

module std.dtl.vector;

/* /////////////////////////////////////////////////////////////////////////////
 * Imports
 */

import std.boxutil;
import std.type.traits;
import std.dtl.common;
import std.dtl.memory;
import std.dtl.utility;
import std.dtl.functions.functions;
import std.dtl.functions.predicates;
import std.dtl.range.categories;
import std.dtl.range.filters;

/* ////////////////////////////////////////////////////////////////////////// */

// Notes:
//
// - length() is used, not size()
// - index_type is a member type
// - class_type is a member type

/* ////////////////////////////////////////////////////////////////////////// */


// These in here only until the bloody linker works 
	extern (C)
	{
		// Functions from the C library.
		void *memcpy(void *, void *, uint);
		void *memmove(void *, void *, uint);
	}

/* ////////////////////////////////////////////////////////////////////////// */

private enum { alloc_quantum = 31 };    // Must be (2^n - 1)

static this()
{
    for(int n = 1; n < 32; ++n)
    {
        if(alloc_quantum == (1 << n) - 1)
        {
            return;
        }
    }

    const int invalid_alloc_quantum = 0;
    assert(invalid_alloc_quantum);
}

/* ////////////////////////////////////////////////////////////////////////// */

template _VectorUtil(T)
{
    class VectorRandomAccessRange
		: public RandomAccessRange!(T)
    {
    public:
        alias   T									value_type;
        alias   VectorRandomAccessRange				class_type;
		alias	std.dtl.common.difference_type		difference_type;
		alias	std.dtl.common.index_type			index_type;
		alias	std.dtl.common.size_type			size_type;

    public:
        /// \brief Constructs a vector from the given array
        this(value_type[] elements)
        {
            m_position  =   0;
            m_elements  =   elements;
        }
        /// \brief Constructs a vector from 
        this(value_type[] elements, index_type position)
        {
            m_position  =   position;
            m_elements  =   elements[position .. elements.length];
        }

	/// \name Range methods
	/// @{
    public:

	/// \name Notional range methods
	/// @{
    public:
		class_type dup()
		{
			return new class_type(m_elements, m_position);
		}

		bool is_open()
		{
			return m_position < m_elements.length;
		}

		value_type current()
		{
			return m_elements[m_position];
		}

		void advance()
        {
            ++m_position;
        }
	/// @}

	/// \name Subscriptable range methods
	/// @{
    public:
		size_type length()
		{
			return m_elements.length - m_position;
		}

		value_type opIndex(index_type index)
		in
		{
		}
		body
		{
			return m_elements[m_position + index];
		}

        void advance(difference_type increment)
		in
		{
		}
		body
        {
            difference_type newPos  =   increment + m_position;

            assert(newPos >= 0);
            assert(newPos <= m_elements.length);

            m_position = newPos;
		}
	/// @}

	/// @}

	/// \name Enumeration
	/// @{
    public:
        int opApply(int delegate(inout value_type element) dg)
        {
            int res = 0;

            foreach(value_type element; m_elements[m_position .. m_elements.length])
            {
                res = dg(element);

                if(0 != res)
                {
                    break;
                }
            }

            return res;
        }
	/// @}

	// Members
    private:
        value_type[]    m_elements;
        index_type      m_position;
    }

} // template _VectorUtil(T)

template Vector(T, B = EmptyBase)
{
    class Vector
        : public BaseSelector!(B).selected_type
    {
    /// \name Types
    /// @{
    public:
		/// The value type
        alias   T																	value_type;
		/// The difference type
		alias	std.dtl.common.difference_type										difference_type;
		/// The index type
		alias	std.dtl.common.index_type											index_type;
		/// The size type
		alias	std.dtl.common.size_type											size_type;
		/// The class type
        alias   Vector																class_type;
		/// The parent class type
		alias	BaseSelector!(B).selected_type										parent_class_type;

		/// The range type
        alias   _VectorUtil!(T).VectorRandomAccessRange								range_type;

version(UsingIterators)
{
        alias   _VectorUtil!(T).VectorIterator!(IteratorDirection.forward)			iterator;
        alias   _VectorUtil!(T).VectorIterator!(IteratorDirection.reverse)			reverse_iterator;
} // version UsingIterators

	public:
		alias	int delegate(in value_type v1, in value_type v2)					compare_delegate_type;
		alias	int function(in value_type v1, in value_type v2)					compare_function_type;
		alias	bool delegate(in value_type value)									match_delegate_type;
		alias	bool function(in value_type value)									match_function_type;
		alias	value_type delegate(in value_type value)							transform_delegate_type;
		alias	value_type function(in value_type value)							transform_function_type;

	private:
		alias FunctionPredicate!(match_function_type, value_type)					match_fn_pred_type;
		alias DelegatePredicate!(match_delegate_type, value_type)					match_dg_pred_type;
		alias FunctionFunction!(transform_function_type, value_type, value_type)	transform_fn_pred_type;
		alias DelegateFunction!(transform_delegate_type, value_type, value_type)	transform_dg_pred_type;

//      alias   RandomAccessInputIterator(T)    
    /// @}

    /// Construction
    /// @{
    public:
        /// \brief Constructs an empty vector
        this()
        {}
        /// \brief Constructs a vector with the given number of elements
        this(size_type dim)
        {
            m_elements = new value_type[m_length = dim];
        }
        /// \brief Constructs a vector with the given number of elements, each of 
        /// which is set to \c value
        this(size_type dim, value_type value)
        {
            m_elements = new value_type[m_length = dim];

            foreach(inout value_type element; m_elements)
            {
                element = value;
            }
        }
        /// \brief Constructs a vector from the given an array
        this(value_type[] values)
        {
            m_elements  =   values.dup;
            m_length    =   m_elements.length;
        }
        /// \brief Constructs a vector as a copy of the given vector \c v
        this(class_type v)
        {
            m_elements	=	v.m_elements[0 .. v.length].dup;
        }
    /// @}

    /// Operations
    /// @{
    public:
        /// \brief Pushes an element onto the vector
        void push_back(in value_type value)
        {
			debug { m_bChangedDuringForeach = true; }

            // Determine whether there is any space in the elements array
            if(!(m_length < m_elements.length))
            {
                int len =   m_elements.length;

                len = (3 * len) / 2;
                len = (1 + len + alloc_quantum) & ~(alloc_quantum);

                m_elements.length   =   len;
            }

            m_elements[m_length++] = value;
        }

        /// \brief Ensures that the vector has the capacity of at least \c n
        void reserve(size_type n)
        {
            if(n < capacity)
            {
                m_elements.length = n;
            }
        }

        /// \brief Empties the vector and releases all storage associated with the previous contents
        void clear()
        {
			debug { m_bChangedDuringForeach = true; }

            m_length = 0;

//            Buffer!(value_type).release(m_elements);
			m_elements = null;
        }

        /// \brief Reverses the contents of the vector
        class_type reverse()
        {
			debug { m_bChangedDuringForeach = true; }

            if(length > 1)
            {
                m_elements[0 .. m_length].reverse;
            }

            return this;
        }

        /// \brief Sorts the contents of the vector
        class_type sort()
        {
			debug { m_bChangedDuringForeach = true; }

            if(length > 1)
            {
                m_elements[0 .. m_length].sort;
            }

            return this;
        }

        template sortWith(F) { final value_type[] sortWith(F f)
        {
			debug { m_bChangedDuringForeach = true; }

            if(length > 1)
            {
                if(f(m_elements[0], m_elements[1]))
                {
                    printf("true\n");
                }
                else
                {
                    printf("false\n");
                }
            }

            return this;
        }}

        /// \brief Erases the element at the given index
        void erase(index_type index)
        {
			debug { m_bChangedDuringForeach = true; }

            erase(index, 1 + index);
        }

        /// \brief Erases the element at the given index
        void erase(index_type from, index_type to)
        in
        {
            assert(from < length);
            assert(to <= length);
        }
        body
        {
			debug { m_bChangedDuringForeach = true; }

            if(to == length)
            {
                m_length = from;
            }
            else
            {
				alias	Memory!(value_type)	Mem_t;	// Have to define as an alias here, otherwise won't link

                // Need to move them down
//                Mem_t.move(&m_elements[from], &m_elements[to], m_length - to);
memmove(&m_elements[from], &m_elements[to], value_type.sizeof * (m_length - to));

                m_length -= (to - from);
            }

            if(0 == length)
            {
                // Completely empty, so decide whether to discard the array
                if(capacity > 128)
                {
//                    Buffer!(value_type).release(m_elements);
					m_elements = null;
                }
                m_length        =   0;
            }
            else if(length * 8 <= capacity)
            {
                // There's a lot of wastes space, so simply drop it by a substantial amount
                m_elements.length = length * 2;
            }
        }
    /// @}

    /// \name Attributes
    /// @{
    public:
        /// Indicates whether the vector is empty
        bool isEmpty()
        {
            return 0 == m_length;
        }

        /// Returns the number of elements in the vector
        size_type length()
        {
            return m_length;
        }

        /// Returns the current capacity of the elements
        size_type capacity()
        {
            return m_elements.length;
        }
    /// @}

    /// \name Operators
    /// @{
    public:
        void opIndexAssign(value_type value, index_type index)
        in
        {
            assert(index < length);
        }
        body
        {
			debug { m_bChangedDuringForeach = true; }

            m_elements[index] = value;
        }

        value_type opIndex(index_type index)
        in
        {
            assert(index < length);
        }
        body
        {
            return m_elements[index];
        }
    /// @}

    /// \name Enumeration #1: foreach
    /// @{
    public:
        int opApply(int delegate(inout value_type element) dg)
        {
            int res = 0;

			debug { m_bChangedDuringForeach = false; }

            foreach(value_type element; m_elements[0 .. length])
            {
                res = dg(element);

				debug { assert(!m_bChangedDuringForeach); }

                if(0 != res)
                {
                    break;
                }
            }

            return res;
        }
    /// @}

    /// \name Enumeration #2: Iterators
    /// @{
    public:
version(UsingIterators)
{
        iterator begin()
        {
			return new range_type(m_elements[0 .. length]);
        }

        iterator end()
        {
			return new range_type(m_elements[length .. length]);
        }
} // version UsingIterators
	/// @}

    /// \name Enumeration #3: Ranges
    /// @{
    public:
        /// \brief Returns a range representing the full
        /// contents of the container
        range_type opSlice()
        {
            return new range_type(m_elements[0 .. length]);
        }

        range_type opSlice(index_type from, index_type to)
        in
        {
            assert(from < length);
            assert(to <= length);
        }
        body
        {
            return new range_type(m_elements[from .. to]);
        }

        /// \brief Returns a range containing the same
        /// number of value_type elements as in the container,
        /// where the resultant values are obtained by
        /// transforming each contained element by applying
        /// the unary function \c f
		TransformedRange!(range_type, transform_fn_pred_type, value_type) transform(transform_function_type d)
		{
			return new TransformedRange!(range_type, transform_fn_pred_type, value_type)(opSlice(), new transform_fn_pred_type(d));
		}
		TransformedRange!(range_type, transform_dg_pred_type, value_type) transform(transform_delegate_type d)
		{
			return new TransformedRange!(range_type, transform_dg_pred_type, value_type)(opSlice(), new transform_dg_pred_type(d));
		}

/+
        /// \brief Returns a range containing the same
        /// number of value_type elements as in the container,
        /// where the resultant values are obtained by
        /// transforming each contained element by applying
        /// the unary function \c f

		template transform(F) { final TransformedRange!(range_type, F) transform(F f = new F())
		{
			return new TransformedRange!(range_type, F)(opSlice(), f);
		}}

        template(F)
        range_type transform(F f);

        /// \brief Indicates whether the contained elements
        /// contain an element for which the unary function
        /// \c f returns true.
        template(F)
        bool detect(F f);
        /// \brief Indicates whether the contained elements
        /// contain an element for which the unary function
        /// \c f returns true. If such an element is found
        /// its value is returned in the given out parameter
        template(F)
        bool detect(F f, out value_type value);
+/

        value_type[] entries()
        {
            value_type[]    e   =   new value_type[cast(int)(length)];
            index_type      i   =   0;

            foreach(value_type v; this)
            {
                e[i++] = v;
            }

            return e;
        }

        /// \brief Returns true if the given value exists in
        /// the container
        bool contains(value_type comperand)
        {
            foreach(value_type v; this)
            {
                if(v == comperand)
                {
                    return true;
                }
            }

            return false;
        }

        /// \brief Returns the minimum element in the container
        /// \note This relies on the value_type being comparable
        value_type  min()
        {
            if(isEmpty())
            {
                // This will need to be genericised in order to work with 
                // reference types and scalar types
                return value_type.min;
            }
            else
            {
                value_type  m = value_type.max;
                foreach(value_type v; this)
                {
                    if(v == value_type.min)
                    {
                        return value_type.min;
                    }
                    else if(v < m)
                    {
                        m = v;
                    }
                }

                return m;
            }
        }
/+
        /// \brief Returns the minimum element in the container
        /// where ordering is determined by the integer value
        /// (<0, 0, >0) provide by the binary function \c f
        template(F)
        value_type  minWith(F f);
+/

        /// \brief Returns the maximum element in the container
        /// \note This relies on the value_type being comparable
        value_type  max()
        {
            if(isEmpty())
            {
                // This will need to be genericised in order to work with 
                // reference types and scalar types
                return value_type.max;
            }
            else
            {
                value_type  m = value_type.min;
                foreach(value_type v; this)
                {
                    if(v == value_type.max)
                    {
                        return value_type.max;
                    }
                    else if(m < v)
                    {
                        m = v;
                    }
                }

                return m;
            }
        }

        /// \brief Returns a range representing all elements
        /// for which the delegate \c d returns true.
        /// \note This is the opposite of select()
		MatchedRange!(range_type, match_fn_pred_type) select(match_function_type d)
		{
			return new MatchedRange!(range_type, match_fn_pred_type)(opSlice(), new match_fn_pred_type(d));
		}
		MatchedRange!(range_type, match_dg_pred_type) select(match_delegate_type d)
		{
			return new MatchedRange!(range_type, match_dg_pred_type)(opSlice(), new match_dg_pred_type(d));
		}
        /// \brief Returns a range representing all elements
        /// for which the function \c f returns true
        /// \note This is the opposite of reject()
		template select0(F) { final MatchedRange!(range_type, F) select0()
		{
			return new MatchedRange!(range_type, F)(opSlice(), new F());
		}}
		template select1(F) { final MatchedRange!(range_type, F) select1(F f)
		{
			return new MatchedRange!(range_type, F)(opSlice(), f);
		}}

/+
        /// \brief Returns the maximum element in the container
        /// where ordering is determined by the integer value
        /// (<0, 0, >0) provide by the binary function \c f
        template(F)
        value_type  maxWith(F f);

        /// \brief Returns a range representing all elements
        /// for which the function \c f returns false.
        /// \note This is the opposite of select()
        template(F)
        range_type  reject(F f);

        /// \brief Returns a range representing all elements
        /// for which the function \c f returns true
        /// \note This is the opposite of reject()
        template(F)
        range_type  select(F f);

+/
        /// \brief Returns a range representing the contained
        /// elements sorted by value
        /// \note This relies on the value_type being comparable
/+
        range_type  sort()
        {
//            return range_type)m_elements[0 .. m_length].sort;
//          m_elements.dup

            return null;
        }
+/

/+
        /// \brief Returns a range representing the contained
        /// elements sorted according to the ordering determined
        /// by the integer value (<0, 0, >0) provide by the binary
        /// function \c f
        template(F)
        range_type  sort(F f);
+/
    /// @}

    /// \name Enumeration #4: IContainer
    /// @{
    public:
		value_type[]	toArray()
		{
			return m_elements[0 .. m_length];
		}

    protected:
		Object[]	toObjectArray()
		{
			return std.boxutil.toObjectArray(toArray());
		}
	/// @}

    /// Debugging
    /// @{
    public:
        void dump()
        {
            printf("Start of dump for:%.*s\n", this.toString());

            foreach(value_type element; m_elements)
            {
                printf("element: %d\n", element);
            }

            printf("End of dump for:%.*s\n", this.toString());
        }
    /// @}

    /// \name Members
    /// @{
    private:
        value_type[]    m_elements;
        size_type       m_length;
		debug
		{
			bool		m_bChangedDuringForeach;
		}
    /// @}

        invariant
        {
            assert(m_length <= m_elements.length);
        }
    }
}

/* ////////////////////////////////////////////////////////////////////////// */
