/* /////////////////////////////////////////////////////////////////////////////
 * File:        std/dtl/stack.d
 *
 * Purpose:     Stack template class.
 *
 * Created      12th March 2004
 * Updated:     18th August 2004
 *
 * www:         http://www.synesis.com.au/software/
 *
 * Copyright (C) 2004 by Matthew Wilson
 *
 * This software is provided 'as-is', without any express or implied warranty.
 * In no event will the authors be held liable for any damages arising from the
 * use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not claim
 * that you wrote the original software. If you use this software in a product,
 * an acknowledgment in the product documentation would be appreciated but is
 * not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must not be
 * misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source distribution. 
 *
 * ////////////////////////////////////////////////////////////////////////// */


/** \file std/dtl/stack.d Stack template class */

/* ////////////////////////////////////////////////////////////////////////// */

module std.dtl.stack;

/* /////////////////////////////////////////////////////////////////////////////
 * Imports
 */

import std.boxutil;
import std.dtl.common;
import std.dtl.range.categories;
import std.dtl.range.ranges;

/* ////////////////////////////////////////////////////////////////////////// */

private enum { alloc_quantum = 31 };    // Must be (2^n - 1)

static this()
{
    for(int n = 1; n < 32; ++n)
    {
        if(alloc_quantum == (1 << n) - 1)
        {
            return;
        }
    }

    const int invalid_alloc_quantum = 0;
    assert(invalid_alloc_quantum);
}

/* ////////////////////////////////////////////////////////////////////////// */

private template _StackUtil(T)
{
    class StackRandomAccessRange
		: public RandomAccessRange!(T)
    {
    public:
        alias   T                           value_type;
        alias   StackRandomAccessRange      class_type;
		alias	std.dtl.common.difference_type	difference_type;
		alias	std.dtl.common.index_type		index_type;
		alias	std.dtl.common.size_type		size_type;

    public:
        this(value_type[] elements)
        {
            m_position  =   0;
            m_elements  =   elements;
        }
        this(value_type[] elements, index_type position)
        {
            m_position  =   position;
            m_elements  =   elements[position .. elements.length];
        }

	/// \name Range methods
	/// @{
    public:
		class_type dup()
		{
			return new class_type(m_elements[m_position .. m_elements.length]);
		}

	/// \name Notional range methods
	/// @{
    public:
		bool is_open()
		{
			return m_position < m_elements.length;
		}

		value_type current()
		{
			return m_elements[m_position];
		}

		void advance()
        {
            ++m_position;
        }
	/// @}

	/// \name Subscriptable range methods
	/// @{
    public:
		size_type length()
		{
			return m_elements.length - m_position;
		}

		value_type opIndex(index_type index)
		in
		{
		}
		body
		{
			return m_elements[m_position + index];
		}

        void advance(difference_type increment)
		in
		{
		}
		body
        {
            difference_type newPos  =   increment + m_position;

            assert(newPos >= 0);
            assert(newPos <= m_elements.length);

            m_position = newPos;
		}
	/// @}

	/// @}

	/// \name Enumeration
	/// @{
    public:
        int opApply(int delegate(inout value_type element) dg)
        {
            int res = 0;

            foreach(value_type element; m_elements[m_position .. m_elements.length])
            {
                res = dg(element);

                if(0 != res)
                {
                    break;
                }
            }

            return res;
        }
	/// @}

/+
        /// \brief Returns a range containing the same
        /// number of value_type elements as in the container,
        /// where the resultant values are obtained by
        /// transforming each contained element by applying
        /// the unary function \c f
        template transform(F) { final value_type[] transform(F f)
        {
        }}

        template(F)
        bool detect(F f);

        template(F)
        bool detect(F f, out value_type value);

        value_type[] entries()

        bool contains(value_type comperand)

        value_type min()

        template min(F) { final value_type min(F f)
        {
		}}

        value_type  max()

        template(F)
        value_type  max(F f);

        template(F)
        range_type  reject(F f);

        template(F)
        range_type  select(F f);

        range_type  sort();

        template(F)
        range_type  sort(F f);
+/

    private:
        value_type[]    m_elements;
        index_type      m_position;
    }
} // template _StackUtil

template Stack(T, B = EmptyBase)
{
    // This stack is implemented on top of the built-in arrays. We maintain bottom and
    // top counters. The initial implementation simply expands the array when pushed,
    // and the bottom advances when popped. A later implementation might do some 
    // intelligent resizing and coallescence.
    class Stack
        : public BaseSelector!(B).selected_type
    {
    /// \name Types
    /// @{
    public:
        alias   T										value_type;
        alias   Stack									class_type;
		alias	std.dtl.common.difference_type				difference_type;
		alias	std.dtl.common.index_type					index_type;
		alias	std.dtl.common.size_type					size_type;

		/// The range type
        alias   _StackUtil!(T).StackRandomAccessRange	range_type;

        alias   BaseSelector!(B).selected_type			parent_class_type;

//      alias   RandomAccessInputIterator(T)    
    /// @}

    /// Construction
    /// @{
    public:
        this()
        {
            m_length    =   0;
            m_elements  =   null;
        }
        this(size_type n)
        {
            m_elements = new value_type[n];
        }
        this(value_type[] values)
        {
            m_elements  =   values.dup;
            m_length    =   m_elements.length;
        }
        this(class_type s)
        out
        {
            assert(s.length == this.length);
        }
        body
        {
            size_type   len =   s.length;

            m_elements  =   s.m_elements[0 .. s.m_length].dup;
            m_length    =   len;
        }
    /// @}

    /// Operations
    /// @{
    public:
        void push(in value_type value)
        in
        {
        }
        out
        {
            assert(value == top());
        }
        body
        {
            // Determine whether there is any space in the elements array
            if(!(m_length < m_elements.length))
            {
                int len =   m_elements.length;

                len = (3 * len) / 2;
                len = (1 + len + alloc_quantum) & ~(alloc_quantum);

                m_elements.length   =   len;
            }

            m_elements[m_length++] = value;
        }

        deprecated void push_back(in value_type value)
        {
            push(value);
        }

        /// Returns the top element
        value_type top()
        in
        {
            assert(0 < length);
        }
        body
        {
            return m_elements[m_length - 1];
        }

        /// Pops the top element
        void pop()
        in
        {
            assert(0 < length);
        }
        body
        {
            --m_length;
            if(0 == m_length)
            {
                // Completely empty, so decide whether to discard the array
                if(capacity > 128)
                {
                    m_elements = null;
                }
                m_length        =   0;
            }
            else if(length * 8 <= capacity)
            {
                // There's a lot of wastes space, so simply drop it by a substantial amount
                m_elements.length = length * 2;
            }
        }

        void reserve(size_type n)
        out
        {
            assert(n <= capacity);
        }
        body
        {
            if(n < capacity)
            {
                m_elements.length = n;
            }
        }

        void clear()
        {
            m_length    =   0;
            m_elements  =   null;
        }
    /// @}

    /// Debugging
    /// @{
    public:
        void dump()
        {
            printf("Start of dump for:%.*s\n", this.toString());

            foreach(value_type element; m_elements)
            {
                printf("element: %d\n", element);
            }

            printf("End of dump for:%.*s\n", this.toString());
        }

    /// @}

    /// Attributes
    /// @{
    public:
        /// Indicates whether the stack is empty
        bool isEmpty()
        {
            return 0 == m_length;
        }

        /// Returns the number of elements in the stack
        size_type length()
        {
            return m_length;
        }

        /// Returns the current capacity of the elements
        size_type capacity()
        {
            return m_elements.length;
        }
    /// @}

    /// \name Enumeration 
    /// @{
    public:
        int opApply(int delegate(inout value_type element) dg)
        {
            int res = 0;

            foreach(value_type element; m_elements[0 .. length])
            {
                res = dg(element);

                if(0 != res)
                {
                    break;
                }
            }

            return res;
        }

        range_type opSlice()
        {
            return new range_type(m_elements);
        }
        range_type opSlice(uint from, uint to)
        in
        {
            assert(from < length);
            assert(to <= length);
        }
        body
        {
            return new range_type(m_elements[from .. to]);
        }
    /// @}

    /// \name Enumeration #4: IContainer
    /// @{
    public:
		value_type[]	toArray()
		{
			return m_elements[0 .. m_length];
		}

    protected:
		Object[]	toObjectArray()
		{
			return std.boxutil.toObjectArray(toArray());
		}
	/// @}

    /// \name Iteration
    /// @{
    public:
        range_type begin()
        {
            return new range_type(m_elements);
        }

        range_type end()
        {
            return new range_type(m_elements[length .. length]);
        }
    /// @}

    /// \name Members
    /// @{
    private:
        value_type[]    m_elements;
        index_type      m_length;
    /// @}

        invariant
        {
            assert(m_length <= m_elements.length);
        }
    }
} // template Stack
