/* /////////////////////////////////////////////////////////////////////////////
 * File:        std/dtl/set.d
 *
 * Purpose:     Set template class.
 *
 * Created      14th March 2004
 * Updated:     18th August 2004
 *
 * www:         http://www.synesis.com.au/software/
 *
 * Copyright (C) 2004 by Matthew Wilson
 *
 * This software is provided 'as-is', without any express or implied warranty.
 * In no event will the authors be held liable for any damages arising from the
 * use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not claim
 * that you wrote the original software. If you use this software in a product,
 * an acknowledgment in the product documentation would be appreciated but is
 * not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must not be
 * misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source distribution. 
 *
 * ////////////////////////////////////////////////////////////////////////// */


/** \file std/dtl/set.d Set template class */

/* ////////////////////////////////////////////////////////////////////////// */

module std.dtl.set;

/* /////////////////////////////////////////////////////////////////////////////
 * Imports
 */

import std.dtl.common;
import std.dtl.exceptions.exceptions;
import std.dtl.functions.predicates;
import std.dtl.range.categories;

/* ////////////////////////////////////////////////////////////////////////// */

private enum { alloc_quantum = 31 };    // Must be (2^n - 1)

static this()
{
    for(int n = 1; n < 32; ++n)
    {
        if(alloc_quantum == (1 << n) - 1)
        {
            return;
        }
    }

    const int invalid_alloc_quantum = 0;
    assert(invalid_alloc_quantum);
}

/* ////////////////////////////////////////////////////////////////////////// */

template Set(T, B = EmptyBase, P = Predicate!(T, T))
{
    class Set
        : public BaseSelector!(B).selected_type
    {
    /// \name Types
    /// @{
    public:
        alias   T								value_type;
        alias   Set								class_type;
		alias	std.dtl.common.difference_type	difference_type;
		alias	std.dtl.common.index_type		index_type;
		alias	std.dtl.common.size_type		size_type;

    private:
        const char[]                            sm_containerName    =   "Set";
	/// @}

    /// \name Construction
    /// @{
    public:
        /// \brief Default constructor
        this()
        {
        }
        /// \brief Constructs a set copying in each element from the given set \c s
        this(class_type s)
        {
            foreach(value_type v; s)
            {
                push_back(v);
            }
        }
    /// @}

    /// Operations
    /// @{
    public:
        void push_back(value_type value)
		{
			m_items[value] = 0;
		}

		void remove(value_type value)
		{
			if(!(value in m_items))
			{
                throw new NotFoundException("Element not in " ~sm_containerName);
			}

			delete m_items[value];
		}

		bool contains(value_type value)
		{
			return value in m_items;
		}
    /// @}

    /// Attributes
    /// @{
    public:
        /// Indicates whether the stack is empty
        bool isEmpty()
        {
            return 0 == m_items.length;
        }

        /// Returns the number of elements in the stack
        size_type length()
        {
            return m_items.length;
        }

        /// Returns the current capacity of the elements
        size_type capacity()
        {
            return m_items.length;
        }
    /// @}

    /// \name Enumeration #1: foreach
    /// @{
    public:
        int opApply(int delegate(inout value_type element) dg)
        {
            int res = 0;

			debug { m_bChangedDuringForeach = false; }

            foreach(value_type v; m_items.keys)
            {
                res = dg(v);

				debug { assert(!m_bChangedDuringForeach); }

                if(0 != res)
                {
                    break;
                }
            }

            return res;
        }
	/// @}

    /// \name Members
    /// @{
    private:
		value_type[int]	m_items;
		debug
		{
			bool		m_bChangedDuringForeach;
		}
    /// @}
    }
}

/* ////////////////////////////////////////////////////////////////////////// */
