/* /////////////////////////////////////////////////////////////////////////////
 * File:        std/dtl/queue.d
 *
 * Purpose:     Queue template class.
 *
 * Created      12th March 2004
 * Updated:     31st July 2004
 *
 * www:         http://www.synesis.com.au/software/
 *
 * Copyright (C) 2004 by Matthew Wilson
 *
 * This software is provided 'as-is', without any express or implied warranty.
 * In no event will the authors be held liable for any damages arising from the
 * use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not claim
 * that you wrote the original software. If you use this software in a product,
 * an acknowledgment in the product documentation would be appreciated but is
 * not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must not be
 * misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source distribution. 
 *
 * ////////////////////////////////////////////////////////////////////////// */


/** \file std/dtl/queue.d Queue template class */

/* ////////////////////////////////////////////////////////////////////////// */

module std.dtl.queue;

/* /////////////////////////////////////////////////////////////////////////////
 * Imports
 */

import std.box;
import std.dtl.common;
import std.dtl.range.categories;

/* ////////////////////////////////////////////////////////////////////////// */

private enum { alloc_quantum = 31 };    // Must be (2^n - 1)

static this()
{
    for(int n = 1; n < 32; ++n)
    {
        if(alloc_quantum == (1 << n) - 1)
        {
            return;
        }
    }

    const int invalid_alloc_quantum = 0;
    assert(invalid_alloc_quantum);
}

/* ////////////////////////////////////////////////////////////////////////// */

template Queue(T, B = EmptyBase)
{
    // This queue is implemented on top of the built-in arrays. We maintain bottom and
    // top counters. The initial implementation simply expands the array when pushed,
    // and the bottom advances when popped. A later implementation might do some 
    // intelligent resizing and coallescence.
    class Queue
        : public BaseSelector!(B).selected_type
    {
    /// \name Types
    /// @{
    public:
        alias   T                           value_type;
        alias   Queue                       class_type;
        alias   std.dtl.common.difference_type	difference_type;
        alias   std.dtl.common.index_type		index_type;
        alias   std.dtl.common.size_type		size_type;
    /// @}

    /// Construction
    /// @{
    public:
        /// \brief Default constructor
        this()
        {
            m_front =   0;
            m_back      =   0;
            m_elements  =   null;
        }
        /// \brief Constructs an empty queue, with an initial capacity \c n
        this(size_type n)
        {
            m_elements = new value_type[n];
        }
        /// \brief Constructs a queue containing the elements of the given array
        this(value_type[] values)
        {
            m_elements  =   values.dup;
            m_front =   0;
            m_back      =   m_elements.length;
        }
        /// \brief Constructs a queue containing a copy of the given queue \c q
        this(class_type q)
        out
        {
            assert(q.length == this.length);
        }
        body
        {
            m_elements  =   q.m_elements[q.m_front .. q.m_back].dup;
            m_front =   0;
            m_back      =   m_elements.length;
        }
    /// @}

    /// Operations
    /// @{
    public:
        /// \brief Pushes an element onto the back of the queue
        void push(in value_type value)
        in
        {
        }
        out
        {

        }
        body
        {
            // Determine whether there is sufficient space in the elements array
            if(!(length < capacity))
            {
                int len =   m_elements.length;

                len = (3 * len) / 2;
                len = (1 + len + alloc_quantum) & ~(alloc_quantum);

                m_elements.length   =   len;
            }

            m_elements[m_back++] = value;
        }

        deprecated void push_back(in value_type value)
        {
            push(value);
        }

        /// Returns the top element
        value_type top()
        in
        {
            assert(0 < length);
        }
        body
        {
            return m_elements[m_front];
        }

        /// Pops the top element
        void pop()
        in
        {
            assert(0 < length);
        }
        body
        {
            ++m_front;
            if(m_back == m_front)
            {
                // Completely empty, so decide whether to discard the array
                if(capacity > 128)
                {
                    m_elements = null;
                }
                m_front =   0;
                m_back  =   0;
            }
            else
            {
                int len =   length;

                if(len * 8 <= capacity)
                {
                    // There's now wasted space, so we have two options:
                    //
                    // 1. If the remaining content can be copied into the start
                    //    of the array, then we do so, and truncate.
                    // 2. If the remaining content cannot be copied into the start
                    //    of the array, then we allocate a new array

                    if(len < m_front)
                    {
                        m_elements[0 .. len] = m_elements[m_front .. m_back];
                        m_elements.length = len;
                        m_front = 0;
                        m_back = len;
                    }
                    else
                    {
                        m_elements = m_elements[m_front .. m_back].dup;
                        m_front = 0;
                        m_back = m_elements.length;
                        assert(len == m_elements.length);
                    }
                }
            }
        }

        void reserve(size_type n)
        out
        {
            assert(n <= capacity);
        }
        body
        {
            if(n < capacity)
            {
                m_elements.length = n;
            }
        }

        void clear()
        {
            m_front = m_back = 0;
            m_elements = null;
        }
    /// @}

    /// Debugging
    /// @{
    public:
        void dump()
        {
            printf("Start of dump for:%.*s\n", this.toString());

            foreach(value_type element; m_elements)
            {
                printf("element: %d\n", element);
            }

            printf("End of dump for:%.*s\n", this.toString());
        }

    /// @}

    /// Attributes
    /// @{
    public:
        /// Indicates whether the queue is empty
        bool isEmpty()
        {
            return m_back == m_front;
        }

        /// Returns the number of elements in the queue
        size_type length()
        {
            return m_back - m_front;
        }

        /// Returns the current capacity of the elements
        size_type capacity()
        {
            return m_elements.length;
        }
    /// @}

    /// \name Enumeration #1: foreach
    /// @{
    public:
        int opApply(int delegate(inout value_type element) dg)
        {
            int res = 0;

            foreach(value_type element; m_elements[m_front .. m_back])
            {
                res = dg(element);

                if(0 != res)
                {
                    break;
                }
            }

            return res;
        }
    /// @}

    /// \name Enumeration #4: IContainer
    /// @{
	public:
		value_type[]	toArray()
		{
			return m_elements[m_front .. m_back];
		}

    protected:
		Object[]	toObjectArray()
		{
			return std.boxutil.toObjectArray(toArray());
		}
	/// @}

    /// \name Members
    /// @{
    private:
        value_type[]    m_elements;
        index_type      m_front;
        index_type      m_back;
    /// @}

        invariant
        {
            assert(m_front <= m_back);
            assert((m_back - m_front) <= m_elements.length);
        }
    }
}

unittest
{
	printf("unittest:Queue\n");
}

/* ////////////////////////////////////////////////////////////////////////// */
