/* /////////////////////////////////////////////////////////////////////////////
 * File:        std/dtl/map.d
 *
 * Purpose:     Map template class.
 *
 * Created      13th March 2004
 * Updated:     20th August 2004
 *
 * www:         http://www.synesis.com.au/software/
 *
 * Copyright (C) 2004 by Matthew Wilson
 *
 * This software is provided 'as-is', without any express or implied warranty.
 * In no event will the authors be held liable for any damages arising from the
 * use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not claim
 * that you wrote the original software. If you use this software in a product,
 * an acknowledgment in the product documentation would be appreciated but is
 * not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must not be
 * misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source distribution. 
 *
 * ////////////////////////////////////////////////////////////////////////// */


/** \file std/dtl/map.d Map template class */

/* ////////////////////////////////////////////////////////////////////////// */

module std.dtl.map;

/* /////////////////////////////////////////////////////////////////////////////
 * Imports
 */

import std.dtl.common;
import std.dtl.utility;
import std.dtl.exceptions.exceptions;
import std.dtl.range.categories;

/* ////////////////////////////////////////////////////////////////////////// */

private enum { alloc_quantum = 31 };    // Must be (2^n - 1)

static this()
{
    for(int n = 1; n < 32; ++n)
    {
        if(alloc_quantum == (1 << n) - 1)
        {
            return;
        }
    }

    const int invalid_alloc_quantum = 0;
    assert(invalid_alloc_quantum);
}

/* ////////////////////////////////////////////////////////////////////////// */

template Pair(K, V)
{
    struct Pair
    {
        alias   K                       key_type;
        alias   V                       value_type;

        key_type    key;
        value_type  value;
    }
}

template Map(K, V)
{
    class Map
    {
    /// \name Types
    /// @{
    public:
        alias   K				            key_type;
        alias   V			                value_type;
        alias   Map		                    class_type;
		alias	std.dtl.common.difference_type	difference_type;
		alias	std.dtl.common.index_type		index_type;
		alias	std.dtl.common.size_type		size_type;
        alias   Pair!(K, V)				    pair_type;
    private:
        alias   value_type[key_type]	    elements_type;
    /// @}

    /// \name Construction
    /// @{
    public:
        /// \brief Default constructor
        this()
        {}
    /// @}

    /// \name Operations
    /// @{
    public:
        /// \brief Sets an element in the array
        void opIndexAssign(value_type value, key_type key)
        {
            m_elements[key] = value;
        }

        /// \brief Returns the value for a given key, or throws an NotFoundException
        value_type opIndex(key_type key)
        {
            if(!(key in m_elements))
            {
                throw new NotFoundException("No element with given key");
            }

            return m_elements[key];
        }

        /// \brief Erases the element corresponding to the given key
        void erase(key_type key)
        {
            if(!(key in m_elements))
            {
                throw new NotFoundException("No element with given key");
			}

            delete m_elements[key];
        }

        /// \brief Empties out the contents
        void clear()
        {
            m_elements = null;
        }

        /// \brief Swaps the contents of the given map with this one
        void swap(class_type c)
        {
			std.dtl.utility.swap!(elements_type)(m_elements, c.m_elements);
        }
    /// @}

    /// \name Attributes
    /// @{
    public:
        /// Indicates whether the map is empty
        bool isEmpty()
        {
            return 0 == m_elements.length;
        }

        /// \brief The number of key-value pairs in the map
        size_type length()
        {
            return m_elements.length;
        }

        /// \brief Returns the current capacity of the elements
        size_type capacity()
        {
            return length;
        }

        /// \brief An array of the keys
        key_type[] keys()
        {
            return m_elements.keys;
        }

        /// \brief An array of the values
        value_type[] values()
        {
            return m_elements.values;
        }

        /// \brief Indicates whether the given key is in the map
        bool containsKey(key_type key)
        {
            return (key in m_elements);
        }

        /// \brief Indicates whether the given key is in the map
        bool containsValue(in value_type value)
        {
            foreach(value_type v; values())
            {
                if(v == value)
                {
                    return true;
                }
            }

            return false;
        }
    /// @}

    /// \name Enumeration
    /// @{
    public:
        int opApply(int delegate(inout key_type key, inout value_type value) dg)
        {
            int             result  =   0;

            foreach(key_type key; m_elements.keys)
            {
// TODO: Find out if this can be done without the lookup
                result = dg(key, m_elements[key]);

                if(0 != result)
                {
                    break;
                }
            }

            return result;
        }

        int opApply(int delegate(inout pair_type entry) dg)
        {
            int             result  =   0;

            foreach(key_type key; m_elements.keys)
            {
                pair_type   pair;

                pair.key    =   key;
                pair.value  =   m_elements[key];

                result = dg(pair);

                if(0 != result)
                {
                    break;
                }
            }

            return result;
        }
	/// @}

    /// \name Members
    /// @{
    private:
        elements_type   m_elements;
    /// @}
    }
}

/* ////////////////////////////////////////////////////////////////////////// */
