/* /////////////////////////////////////////////////////////////////////////////
 * File:        std/dtl/list.d
 *
 * Purpose:     List template class.
 *
 * Created      14th February 2004
 * Updated:     22nd August 2004
 *
 * www:         http://www.synesis.com.au/software/
 *
 * Copyright (C) 2004 by Matthew Wilson
 *
 * This software is provided 'as-is', without any express or implied warranty.
 * In no event will the authors be held liable for any damages arising from the
 * use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not claim
 * that you wrote the original software. If you use this software in a product,
 * an acknowledgment in the product documentation would be appreciated but is
 * not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must not be
 * misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source distribution. 
 *
 * ////////////////////////////////////////////////////////////////////////// */


/** \file std/dtl/list.d List template class */

/* ////////////////////////////////////////////////////////////////////////// */

module std.dtl.list;

/* /////////////////////////////////////////////////////////////////////////////
 * Imports
 */

import std.boxutil;
import std.type.traits;
import std.dtl.common;
import std.dtl.exceptions.exceptions;
import std.dtl.utility;
import std.dtl.functions.functions;
import std.dtl.functions.predicates;
import std.dtl.range.categories;
import std.dtl.range.filters;

/* ////////////////////////////////////////////////////////////////////////// */

// 1. foreach       -   
// 2. Iterators     -   
// 3. Ranges        -   
// 4. Param-Poly    -   

/* ////////////////////////////////////////////////////////////////////////// */

private enum { alloc_quantum = 31 };    // Must be (2^n - 1)

static this()
{
    for(int n = 1; n < 32; ++n)
    {
        if(alloc_quantum == (1 << n) - 1)
        {
            return;
        }
    }

    const int invalid_alloc_quantum = 0;
    assert(invalid_alloc_quantum);
}

/* ////////////////////////////////////////////////////////////////////////// */

template _ListUtil(T)
{
    /// \brief The ListNode
    private class ListNode
    {
    public:
        alias   T							value_type; /* The value */
        alias   ListNode					node_type;  /* . */
        alias   ListNode					class_type; /* . */
		alias	std.dtl.common.difference_type	difference_type;
		alias	std.dtl.common.index_type		index_type;
		alias	std.dtl.common.size_type		size_type;

    public:
        value_type          m_value;    /* The value. */
        node_type           m_previous; /* The previous element in the list. */
        node_type           m_next;     /* The next element in the list. */

    public:
        /// \brief Constructs a solitary node from a value only.
        this(value_type value)
        {
            m_value     =   value;
            m_previous  =   null;
            m_next      =   null;
        }
        /// \brief Constructs a linked node from a value and previous and next nodes.
        this(value_type value, node_type previous, node_type next)
        {
            m_value     =   value;
            m_previous  =   previous;
            m_next      =   next;
        }

    public:
        /// \brief Counts the number of nodes forward from the current node.
        static size_type countLength_forward(node_type node)
        {
            size_type   len;

            for(len = 0; null !== node; node = node.m_next, ++len)
            {}

            return len;
        }

    private:
        invariant
        {
            assert(null === m_previous || m_previous !== m_next);
        }
    }


version(UseIterators)
{
    template ListIteratorAdvancer(IteratorDirection D)
    {
        ListNode advanceNode(ListNode node);
    }

    template ListIteratorAdvancer(IteratorDirection D : IteratorDirection.forward)
    {
        ListNode advanceNode(ListNode node)
        {
            return node.m_next;
        }
    }

    template ListIteratorAdvancer(IteratorDirection D : IteratorDirection.reverse)
    {
        ListNode advanceNode(ListNode node)
        {
            return node.m_previous;
        }
    }

    template ListIterator(IteratorDirection D)
    {
        public class ListIterator
        {
        public:
            alias   T               value_type; /* The value */
            alias   ListIterator    class_type;
        private:
            alias   ListNode        node_type;

        private:
            this(node_type node)
            {
                m_node = node;
            }
            this()
            {
                m_node = null;
            }

        public:
            void next()
            in
            {
                assert(null !== m_node);
            }
            body
            {
                m_node = ListIteratorAdvancer!(D).advanceNode(m_node);
            }

            value_type value()
            in
            {
                assert(null !== m_node);
            }
            body
            {
                return m_node.m_value;
            }

            int opEquals(class_type rhs)
            in
            {
                assert(null !== this);
                assert(null !== rhs);
            }
            body
            {
                return m_node === rhs.m_node;
            }

        private:
            node_type   m_node;
        }
    }
} // version(UseIterators)

    public class ListRange
		: public IterableRangeTag
    {
    public:
        alias   T           value_type;
        alias   ListRange   class_type;
        alias   class_type  range_type;
    private:
        alias   ListNode    node_type;

    private:
        this(node_type from, node_type to)
        {
            m_from  =   from;
            m_to    =   to;
        }

    public:
		class_type dup()
		{
			return new class_type(m_from, m_to);
		}

        bool is_open()
        {
            return m_from !== m_to;
        }

        value_type current()
        in
        {
            assert(m_from !== m_to);
        }
        body
        {
            return m_from.m_value;
        }

        void advance()
        in
        {
            assert(m_from !== m_to);
        }
        body
        {
            m_from = m_from.m_next;
        }

    public:
        int opApply(int delegate(inout value_type element) dg)
        {
            int res = 0;

            for(node_type node = m_from; node !== m_to; node = node.m_next)
            {
                res = dg(node.m_value);

                if(0 != res)
                {
                    break;
                }
            }

            return res;
        }

    private:
        node_type   m_from;
        node_type   m_to;
    };

} // template _ListUtil(T)

template List(T, B = EmptyBase)
{
    /// \brief This class represents a doubly-linked list
    public class List
        : public BaseSelector!(B).selected_type
/+ +Mixins+
version(Mixins)
{
        , mixes Ranges#(List)
}
+/
    {
    /// \name Types
    /// @{
    public:
        alias   T                                                       value_type;
        alias   List                                                    class_type;
		alias	std.dtl.common.difference_type							difference_type;
		alias	std.dtl.common.index_type								index_type;
		alias	std.dtl.common.size_type								size_type;
version(UseIterators)
{
        alias   _ListUtil!(T).ListIterator!(IteratorDirection.forward)  iterator;
        alias   _ListUtil!(T).ListIterator!(IteratorDirection.reverse)  reverse_iterator;
} // version(UseIterators)
        alias   _ListUtil!(T).ListRange                                 range_type;
    private:
        alias   BaseSelector!(B).selected_type                          parent_class_type;
        alias   _ListUtil!(T).ListNode                                  node_type;

	public:
		alias	int delegate(in value_type v1, in value_type v2)		compare_delegate_type;
		alias	int function(in value_type v1, in value_type v2)		compare_function_type;
		alias	bool delegate(in value_type value)						match_delegate_type;
		alias	bool function(in value_type value)						match_function_type;
		alias	value_type delegate(in value_type value)				transform_delegate_type;
		alias	value_type function(in value_type value)				transform_function_type;

	private:
		alias FunctionPredicate!(match_function_type, value_type)					match_fn_pred_type;
		alias DelegatePredicate!(match_delegate_type, value_type)					match_dg_pred_type;
		alias FunctionFunction!(transform_function_type, value_type, value_type)	transform_fn_pred_type;
		alias DelegateFunction!(transform_delegate_type, value_type, value_type)	transform_dg_pred_type;
    /// @}

    private:
        const char[]                                                    sm_containerName    =   "List";

    /// \name Construction
    /// @{
    public:
        /// \brief Default constructor
        this()
        {}
        /// \brief Constructs a list copying in each element from the given list \c l
        this(class_type l)
        {
            foreach(value_type v; l)
            {
                push_back(v);
            }
        }
    /// @}

    /// Operations
    /// @{
    public:
        /// \brief Adds an elements onto the front of the list
        void push_front(value_type value)
        {
			debug { m_bChangedDuringForeach = true; }

            if(null === m_head)
            {
                m_head = m_tail = new node_type(value);
            }
            else
            {
                // Create a node that points back to null, and forward to 
                // what the head currently points to.
                node_type node = new node_type(value, null, m_head);

                // Point the previous head
                m_head.m_previous   =   node;
                m_head              =   node;
            }

            ++m_length;
        }

        /// \brief Adds an elements onto the back of the list
        void push_back(value_type value)
        {
			debug { m_bChangedDuringForeach = true; }

            if(null === m_tail)
            {
                m_head = m_tail = new node_type(value);
            }
            else
            {
                // Create a node that points forward to null, and back to 
                // what the tail currently points to.
                node_type node = new node_type(value, m_tail, null);

                // Point the previous head
                m_tail.m_next   =   node;
                m_tail          =   node;
            }

            ++m_length;
        }

        /// \brief Removes the element at the front of the list
        void pop_front()
        in
        {
            assert(0 != length);
        }
        body
        {
			debug { m_bChangedDuringForeach = true; }

            node_type   new_head = m_head.m_next;

            if(null !== new_head)
            {
                new_head.m_previous = null;
                m_head.m_next = null; // Must do this to avoid dead cycles in GC
                m_head = new_head;
            }
            else
            {
                m_head = m_tail = null;
            }

            --m_length;
        }

        /// \brief Removes the element at the back of the list
        void pop_back()
        in
        {
            assert(0 != length);
        }
        body
        {
			debug { m_bChangedDuringForeach = true; }

            node_type   new_tail = m_tail.m_previous;

            if(null !== new_tail)
            {
                new_tail.m_next     =   null;
                m_tail.m_previous   =   null; // Must do this to avoid dead cycles in GC
                m_tail              =   new_tail;
            }
            else
            {
                m_head = m_tail = null;
            }

            --m_length;
        }

        /// \brief Removes all the elements from the list
        void clear()
        {
			debug { m_bChangedDuringForeach = true; }

            // Clear all the nodes for the GC
            for(node_type node = m_head; null !== node; )
            {
                node_type node_to_clear =   node;

                node = node.m_next;

                node_to_clear.m_previous    =   null;
                node_to_clear.m_next        =   null;
            }
            m_head      =   null;
            m_tail      =   null;
            m_length    =   0;
        }

        /// \brief Reverses the contents of the list in place
        void reverse()
        {
			debug { m_bChangedDuringForeach = true; }

            // (i) Swap the m_next and m_previous in each item.
            for(node_type node = m_head; null !== node;)
            {
				node_type		t               =   node.m_previous;
								node.m_previous	=   node.m_next;
								node.m_next		=   t;
/+
                swap!(node_type)(node.m_previous, node.m_next);
+/

                node = node.m_previous; // Now swapped!!
            }

            // (ii) Swap the m_head and m_tail
			node_type		t       =   m_head;
							m_head	=   m_tail;
							m_tail	=   t;
/+
            swap!(node_type)(m_head, m_tail);
+/
        }
    /// @}

    /// Attributes
    /// @{
    public:
        /// Indicates whether the stack is empty
        bool isEmpty()
        {
            return 0 == m_length;
        }

        /// Returns the number of elements in the stack
        size_type length()
        {
            return m_length;
        }

        /// Returns the current capacity of the elements
        size_type capacity()
        {
            return m_length;
        }

        /// \brief Returns the value at the head of the list 
        value_type front()
        in
        {
            assert(0 != length);
        }
        body
        {
            return m_head.m_value;
        }

        /// \brief Returns the value at the tail of the list 
        value_type back()
        in
        {
            assert(0 != length);
        }
        body
        {
            return m_tail.m_value;
        }
    /// @}

    /// \name Enumeration #1: foreach
    /// @{
    public:
        int opApply(int delegate(inout value_type element) dg)
        {
            int res = 0;

			debug { m_bChangedDuringForeach = false; }

            for(node_type node = m_head; null !== node; node = node.m_next)
            {
                res = dg(node.m_value);

				debug { assert(!m_bChangedDuringForeach); }

                if(0 != res)
                {
                    break;
                }
            }

            return res;
        }
    /// @}

version(UseIterators)
{
    /// \name Enumeration #2: Iterators
    /// @{
    public:
        /// Returns an iterator to the start of the contained sequence
        iterator begin()
        {
            return new iterator(m_head);
        }

        /// Returns an iterator to the stop of the contained sequence
        iterator end()
        {
            return new iterator();
        }

        /// Returns an iterator to the start of the contained reverse sequence
        reverse_iterator rbegin()
        {
            return new reverse_iterator(m_tail);
        }

        /// Returns an iterator to the end of the contained reverse sequence
        reverse_iterator rend()
        {
            return new reverse_iterator();
        }
    /// @}
} // version(UseIterators)

    /// \name Enumeration #3: Ranges
    /// @{
    public:
        /// \brief Returns a range representing the full
        /// contents of the container
        range_type opSlice()
        {
            return new range_type(m_head, null);
        }

        /// \brief Returns a range containing the same
        /// number of value_type elements as in the container,
        /// where the resultant values are obtained by
        /// transforming each contained element by applying
        /// the unary function \c f
		TransformedRange!(range_type, transform_fn_pred_type, value_type) transform(transform_function_type d)
		{
			return new TransformedRange!(range_type, transform_fn_pred_type, value_type)(opSlice(), new transform_fn_pred_type(d));
		}
		TransformedRange!(range_type, transform_dg_pred_type, value_type) transform(transform_delegate_type d)
		{
			return new TransformedRange!(range_type, transform_dg_pred_type, value_type)(opSlice(), new transform_dg_pred_type(d));
		}

        /// \brief Returns a range containing the same
        /// number of value_type elements as in the container,
        /// where the resultant values are obtained by
        /// transforming each contained element by applying
        /// the unary function \c f
        template transform0(F, T2 = value_type) { final TransformedRange!(range_type, F, T2) transform0()
        {
            return new TransformedRange!(range_type, F, T2)(opSlice(), new F());
        }}
        template transform1(F, T2 = value_type) { final TransformedRange!(range_type, F, T2) transform1(F f = new F())
        {
            return new TransformedRange!(range_type, F, T2)(opSlice(), f);
        }}

		bool detect(value_type value)
		{
			foreach(value_type v; this)
			{
				if(v == value)
				{
					return true;
				}
			}

			return false;
		}

		static private value_type	sm_nullReceiver;

        /// \brief Indicates whether the contained elements
        /// contain an element for which the unary function
        /// \c f returns true.
		template detectWith(F) { final bool detectWith(F f)
		{
			foreach(value_type v; this)
			{
				if(f(v))
				{
					return true;
				}
			}

			return false;
		}}
        /// \brief Indicates whether the contained elements
        /// contain an element for which the unary function
        /// \c f returns true. If such an element is found
        /// its value is returned in the given out parameter
/+
        template detect(F) { final bool detect(F f, out value_type value)
		{
			foreach(value_type v; this)
			{
				if(f(v))
				{
					value = v;

					return true;
				}
			}

			return false;
		}}
+/

        /// \brief Returns an array of the contained items
        /// \note This is never an array slice, always an array
        range_type entries()
        {
			return opSlice();
        }

        /// \brief Returns true if the given value exists in
        /// the container
        bool contains(value_type comperand)
        {
            foreach(value_type v; this)
            {
                if(v == comperand)
                {
                    return true;
                }
            }

            return false;
        }

        /// \brief Returns the minimum element in the container
        /// \note This relies on the value_type being comparable
        /// \note Throws an instance of NotFoundException if the
        /// container is empty
        value_type min()
        {
            if(isEmpty())
            {
                // This will need to be genericised in order to work with 
                // reference types and scalar types
                throw new NotFoundException(sm_containerName ~ " container is empty");
            }
            else
            {
                value_type  m = front;

				if(!std.type.traits.isMinValue(m))
				{
					foreach(value_type v; this)
					{
						if(v < m)
						{
							m = v;

							if(std.type.traits.isMinValue(m))
							{
								break;
							}
						}
					}
				}

                return m;
            }
        }
        /// \brief Returns the minimum element in the container
        /// where ordering is determined by the integer value
        /// (<0, 0, >0) provide by the binary function \c f
        template minWith(F) { final value_type minWith(F f)
        {
            if(isEmpty())
            {
                // This will need to be genericised in order to work with 
                // reference types and scalar types
                throw new NotFoundException(sm_containerName ~ " container is empty");
            }
            else
            {
                value_type  m = front;

                foreach(value_type v; this)
                {
					if(f(v, m))
                    {
                        m = v;
                    }
                }

                return m;
            }
        }}

        /// \brief Returns the maximum element in the container
        /// \note This relies on the value_type being comparable
        value_type  max()
        {
            if(isEmpty())
            {
                // This will need to be genericised in order to work with 
                // reference types and scalar types
                throw new NotFoundException(sm_containerName ~ " container is empty");
            }
            else
            {
                value_type  m = front;

				if(!std.type.traits.isMaxValue(m))
				{
					foreach(value_type v; this)
					{
						if(m < v)
						{
							m = v;

							if(std.type.traits.isMaxValue(m))
							{
								break;
							}
						}
					}
				}

                return m;
            }
        }

        /// \brief Returns the maximum element in the container
        /// where ordering is determined by the integer value
        /// (<0, 0, >0) provide by the binary function \c f
        template maxWith(F) { final value_type maxWith(F f)
        {
            if(isEmpty())
            {
                // This will need to be genericised in order to work with 
                // reference types and scalar types
                throw new NotFoundException(sm_containerName ~ " container is empty");
            }
            else
            {
                value_type  m = front;

                foreach(value_type v; this)
                {
					if(!f(v, m))
                    {
                        m = v;
                    }
                }

                return m;
            }
        }}

        /// \brief Returns a range representing all elements
        /// for which the delegate \c d returns false.
        /// \note This is the opposite of select()
        MatchedRange!(range_type, Not!(match_fn_pred_type)) reject(match_function_type d)
        {
            return new MatchedRange!(range_type, Not!(match_fn_pred_type))(opSlice(), new Not!(match_fn_pred_type)(new match_fn_pred_type(d)));
        }

        MatchedRange!(range_type, Not!(match_dg_pred_type)) reject(match_delegate_type d)
        {
            return new MatchedRange!(range_type, Not!(match_dg_pred_type))(opSlice(), new Not!(match_dg_pred_type)(new match_dg_pred_type(d)));
        }

        /// \brief Returns a range representing all elements
        /// for which the predicate \c p returns false.
        /// \note This is the opposite of select()
        template reject0(P) { final MatchedRange!(range_type, Not!(P)) reject0()
        {
            return new MatchedRange!(range_type, Not!(P))(opSlice(), new Not!(P)(new P()));
        }}
        template reject1(P) { final MatchedRange!(range_type, Not!(P)) reject1(P p)
        {
            return new MatchedRange!(range_type, Not!(P))(opSlice(), new Not!(P)(p));
        }}

        /// \brief Returns a range representing all elements
        /// for which the delegate \c d returns true.
        /// \note This is the opposite of select()
		MatchedRange!(range_type, match_fn_pred_type) select(match_function_type d)
		{
			return new MatchedRange!(range_type, match_fn_pred_type)(opSlice(), new match_fn_pred_type(d));
		}
		MatchedRange!(range_type, match_dg_pred_type) select(match_delegate_type d)
		{
			return new MatchedRange!(range_type, match_dg_pred_type)(opSlice(), new match_dg_pred_type(d));
		}
        /// \brief Returns a range representing all elements
        /// for which the function \c f returns true
        /// \note This is the opposite of reject()
		template select0(F) { final MatchedRange!(range_type, F) select0()
		{
			return new MatchedRange!(range_type, F)(opSlice(), new F());
		}}
		template select1(F) { final MatchedRange!(range_type, F) select1(F f)
		{
			return new MatchedRange!(range_type, F)(opSlice(), f);
		}}

/+
        /// \brief Returns a range representing the contained
        /// elements sorted by value
        /// \note This relies on the value_type being comparable
        range_type  sort();
        /// \brief Returns a range representing the contained
        /// elements sorted according to the ordering determined
        /// by the integer value (<0, 0, >0) provide by the binary
        /// function \c f
        template(F)
        range_type  sortWith(F f);
+/
    /// @}

    /// \name Enumeration #4: IContainer
    /// @{
	public:
		value_type[]	toArray()
		{
			value_type[]	values;

			values = new value_type[length];
			values.length = 0;

			foreach(value_type v; this)
			{
				values ~= v;
			}

			return values;
		}

    protected:
		Object[]	toObjectArray()
		{
			return std.boxutil.toObjectArray(toArray());
		}
	/// @}

    /// \name Members
    /// @{
    private:
        node_type       m_head;
        node_type       m_tail;
        size_type       m_length;
		debug
		{
			bool		m_bChangedDuringForeach;
		}
    /// @}

        invariant
        {
//            assert((null === m_head) == (null === m_tail));
//            assert((null === m_head) || (null === m_head.m_previous));
//            assert((null === m_tail) || (null === m_tail.m_next));
            size_type l = node_type.countLength_forward(m_head);

            if(l != m_length)
            {
                printf("m_length (%d) != counted length (%u)\n", m_length, l);
                assert(l == m_length);
            }
        }
    }
} // template List(T)


/* ////////////////////////////////////////////////////////////////////////// */
